/*

An illustration for the OpenFlexure Microscope; how to put the nut in

(c) 2016 Richard Bowman - released under CERN Open Hardware License

*/

use <../openscad/libs/compact_nut_seat.scad>
use <../openscad/libs/utilities.scad>
use <../openscad/libs/lib_actuator_assembly_tools.scad>
use <../openscad/libs/microscope_parameters.scad>
use <../openscad/libs/libfeet.scad>
use <../openscad/libs/gears.scad>
use <../openscad/libs/main_body_structure.scad>
use <../openscad/libs/libdict.scad>
use <../openscad/libs/z_axis.scad>

use <librender/hardware.scad>
use <librender/render_settings.scad>
use <librender/assembly_parameters.scad>
use <librender/render_utils.scad>
use <actuator_assembly.scad>
use <librender/rendered_main_body.scad>

FRAME = 3;
MANUAL = false;
UP = 1; // actuator up 1, centre 0, down -1, full up or down (extra 0.4mm clearance) +10 and -10
AXIS_X = true; // true is X, false is Z


$vpr = [90,0,270];
render_band_insertion(band_insertion_frame_parameters(FRAME), manual=MANUAL, up=UP, axis_x=AXIS_X);

module cut_actuator_housing(params, cut=true, axis_x=true){
    module housing(params, axis_x){
        if (axis_x){
            xy_screw_seat(params, label="X");
        }
        else {
            translate_y(-z_nut_y(params)){
                z_actuator_housing(params);
            }
        }
    }

    difference(){
        housing(params, axis_x);
        // cutout actuator hole
        difference(){
            translate([-3,-10,0]){
                cube([6,10,5]);
            }
            actuator_end_cutout();
        }
        // only render half
        if (cut) {
            rotate_y(-90){
                cylinder(r=99,h=99,$fn=4);
            }
        }
    }
}

module render_band_insertion(frame_dict, manual=false, up=0, axis_x=true){
    function no_lug_params() = let(
        params = default_params()
    ) replace_value("include_motor_lugs", false, params);    
    params = !manual ? default_params() : no_lug_params();
    actuator_params = replace_value("print_ties", false, params);

    foot_tr = key_lookup("foot_tr", frame_dict);
    band_tr = key_lookup("band_tr", frame_dict);
    tool_tr = key_lookup("tool_tr", frame_dict);
    tool_apart = key_lookup("tool_apart", frame_dict);
    casing_cut = key_lookup("casing_cut", frame_dict);
    casing_alpha = key_lookup("casing_alpha", frame_dict);
    foot_alpha = key_lookup("foot_alpha", frame_dict);
    tool_kink = key_lookup("tool_kink", frame_dict);
    band_stretch = key_lookup("band_stretch", frame_dict);
    nut_tool = key_lookup("nut_tool", frame_dict);
    actuator_h = key_lookup("actuator_h", params);

    echo("xy_actuator_travel",xy_actuator_travel(params));
    echo("xy_stage_motion up", (xy_actuator_travel(params))*7/4);
    echo("xy_stage_motion down", (xy_actuator_travel(params)+0*0.5)*7/4);

    echo("z_actuator_travel",z_actuator_travel(params));
    echo("z_stage_motion up", (z_actuator_travel(params))*1.0);
    echo("z_stage_motion down", (z_actuator_travel(params)+0*0.5)*1.0);

    if (axis_x){
        tt = ((up<5) && (up>-5)) ? up*(xy_actuator_travel(params)) : sign(up)*(xy_actuator_travel(params)+actuator_travel_clearance());
        nub = xy_actuator_top_nub(params);
        translate_z(tt){
            coloured_render("orange", 1.0){
                xy_actuator_column(actuator_params);
            }
        }
    }

    
    if (!axis_x){
        th = z_actuator_tilt(params);
        echo("z tilt", z_actuator_tilt(params));
        tt = ((up<5) && (up>-5)) ? up*(z_actuator_travel(params)) : sign(up)*(z_actuator_travel(params)+actuator_travel_clearance());
        t_z = cos(th)*tt;
        t_y = -sin(th)*tt;
        translate([0,t_y,t_z]){
            translate_y(-z_nut_y(params)){
                coloured_render("orange", 1.0){
                    z_actuator_column(actuator_params);
                }
            }
        }
    }

 
    if ((axis_x) && (up>0.5)){
        extra_stretch = (up>5)? 0.4 : 0;
        band_h = actuator_h + 0*band_stretch + xy_actuator_travel(params) + extra_stretch;
        translate(band_tr){
            viton_band_in_situ_vertical(h=band_h, tool_kink=tool_kink);
        }
    }

    color(extras_colour(), 1){
        difference(){
            translate(foot_tr){
                render(6){
                    if (axis_x){
                        outer_foot(params, lie_flat=false, letter="X");
                    }
                    else {
                        middle_foot(params, lie_flat=false, letter="z");
                    }
                }
            }
            rotate_y(-90){
                cylinder(r=99,h=99,$fn=4);
            }
        }
    }

    // See though object last
    color(body_colour(), 0.5){
        render(6){
            cut_actuator_housing(params, cut=casing_cut, axis_x=axis_x);
        }
    }
}


/* The arms of the band tool for rendering with option flex the prongs apart.

To simulate flexing if apart is true this repeates the translations and rotations
used in band_tool_arms(vertical=true), but a rotation slightly below 90
it also corrects so that the effective rotation is about an axis on the plane at the top
of the holder.

:param params: The microscope parameters dictionary.
:param apart: Set to true for the prongs to be sprung apart.
*/
module band_tool_arms_render(params, apart=false){
    if (!apart){
        band_tool_arms(params, vertical=true);
    }
    else{
        h=band_tool_end_thickness();
        angle = 88;
        // effective centre of rotation shift
        shift = holder_height()-2;
        reflect_y(){
            translate_y(band_tool_arm_sep()/2 - shift*cos(angle)){
                rotate_x(angle){
                    band_tool_arm_with_end(params, h);
                }
            }
        }
    }
}

function band_insertion_frame_parameters(frame_number) = let(
    band_elongate = -9,
    frame1 = [["foot_tr", [0,0,-40-band_elongate]],
              ["band_tr", [0,0,-40-band_elongate]],
              ["tool_tr", [0,0,-40]],
              ["tool_apart", false],
              ["casing_cut", false],
              ["casing_alpha", 1],
              ["foot_alpha", 1],
              ["tool_kink", true],
              ["band_stretch", band_elongate],
              ["nut_tool", true]],

    frame2 = [["foot_tr", [0,0,0]],
              ["band_tr", [0,0,0]],
              ["tool_tr", [0,0,0+band_elongate]],
              ["tool_apart", false],
              ["casing_cut", false],
              ["casing_alpha", 1],
              ["foot_alpha", 1],
              ["tool_kink", true],
              ["band_stretch", band_elongate],
              ["nut_tool", true]],

    frame3 = [["foot_tr", [0,0,0]],
              ["band_tr", [0,0,0]],
              ["tool_tr", [0,0,0+band_elongate]],
              ["tool_apart", false],
              ["casing_cut", true],
              ["casing_alpha", .5],
              ["foot_alpha", .5],
              ["tool_kink", true],
              ["band_stretch", band_elongate],
              ["nut_tool", true]],

    frame4 = [["foot_tr", [0,0,0]],
              ["band_tr", [0,0,0]],
              ["tool_tr", [0,0,0]],
              ["tool_apart", true],
              ["casing_cut", true],
              ["casing_alpha", .5],
              ["foot_alpha", .5],
              ["tool_kink", true],
              ["band_stretch", 0],
              ["nut_tool", true]],

    frame5 = [["foot_tr", [0,0,0]],
              ["band_tr", [0,0,0]],
              ["tool_tr", [0,0,-40]],
              ["tool_apart", false],
              ["casing_cut", true],
              ["casing_alpha", .5],
              ["foot_alpha", .5],
              ["tool_kink", false],
              ["band_stretch", 0],
              ["nut_tool", true]],

    frame6 = [["foot_tr", [0,0,0]],
              ["band_tr", [0,0,0]],
              ["tool_tr", [0,0,-40]],
              ["tool_apart", false],
              ["casing_cut", false],
              ["casing_alpha", 1],
              ["foot_alpha", 1],
              ["tool_kink", false],
              ["band_stretch", 0],
              ["nut_tool", false]],

    frames = [frame1, frame2, frame3, frame4, frame5, frame6]
) frames[frame_number-1];
